# Fibonacci — Recursion vs Dynamic Programming (Java)

This mini-project introduces the Fibonacci sequence and contrasts three ways to compute `fib(n)`: a naive recursive solution, a top-down dynamic-programming (memoization) approach, and a bottom-up dynamic-programming approach. A final extension invites you to support arbitrarily large `n` using `BigInteger`.

---

## 1) The Fibonacci problem (brief)

The Fibonacci sequence is defined by:

- `fib(0) = 0`, `fib(1) = 1`
- `fib(n) = fib(n-1) + fib(n-2)` for `n ≥ 2`

Goal: given an integer `n`, compute `fib(n)`.

**Overflow note:** `long` overflows at `fib(93)` (since `fib(93) > Long.MAX_VALUE`). For large `n`, use `BigInteger` (see Section 5).

---

## 2) Recursive solution (naive)

**Idea:** Translate the mathematical recurrence directly.

```java
private static long fibRec(int n) {
    if (n == 0 || n == 1) return n;
    return fibRec(n - 1) + fibRec(n - 2);
}
```

- **Time complexity:** `O(2^n)`
- **Space complexity:** `O(n)` (call stack depth)
- **Limitations:** Massive recomputation of subproblems → very slow for even moderate `n` (e.g., `n ≈ 45+`).

---

## 3) Dynamic Programming — Top-Down (memoization)

**Idea:** Cache results of subproblems to avoid recomputation.

```java
private static long fibDPTopDown(int n, long[] cache) {
    if (cache[n] != 0) return cache[n];
    long ans = (n == 0 || n == 1) ? n
              : fibDPTopDown(n - 1, cache) + fibDPTopDown(n - 2, cache);
    cache[n] = ans;
    return ans;
}
```

- **Time complexity:** `O(n)` (each `fib(k)` computed once)
- **Space complexity:** `O(n)` (memo array) + `O(n)` stack → `O(n)` overall
- **Pros:** Simple upgrade from recursion; preserves recursive structure.
- **Cons:** still limited by recursion depth (`n` around thousands is okay, but not millions).

---

## 4) Dynamic Programming — Bottom-Up (tabulation)

**Idea:** Build solutions from base cases up to `n`. The memory-optimized variant stores only the last two values.

```java
private static long fibDPBottomUp(int n) {
    long prevprev = 0, prev = 1, current = n;
    for (int i = 2; i <= n; i++) {
        current = prevprev + prev;
        prevprev = prev;
        prev = current;
    }
    return current;
}
```

- **Time complexity:** `O(n)`
- **Space complexity:** `O(1)` (with rolling variables).

---

## 5) Extension — BigInteger version (arbitrary precision)

**Task:** Implement a BigInteger version so results are correct beyond `n = 92`.

**Suggested bottom-up implementation:**

```java
import java.math.BigInteger;

private static BigInteger fibBigBottomUp(int n) {
    if (n == 0) return BigInteger.ZERO;
    if (n == 1) return BigInteger.ONE;

    BigInteger prevprev = BigInteger.ZERO; // fib(0)
    BigInteger prev     = BigInteger.ONE;  // fib(1)
    BigInteger current  = BigInteger.ZERO;

    for (int i = 2; i <= n; i++) {
        current = prevprev.add(prev);
        prevprev = prev;
        prev = current;
    }
    return current;
}
```

**Optional challenges:**

- Add a top-down `BigInteger` memoized version.
- Print the number of digits of `fib(n)` (via `toString().length()`).
